use strict;
use utf8;
use IPC::Open3;
use POSIX 'sys_wait_h';
use Schulkonsole::Wrapper;
use Schulkonsole::Error::Error;
use Schulkonsole::Error::DruckquotasError;
use Schulkonsole::Config;
use Schulkonsole::DruckquotasConfig;

package Schulkonsole::Druckquotas;

=head1 NAME
Schulkonsole::Druckquotas - interface to Pykota Druckquotas
=head1 SYNOPSIS
 use Schulkonsole::Druckquotas;
 use Schulkonsole::Session;

=head1 DESCRIPTION

If a wrapper command fails, it usually dies with a Schulkonsole::DruckquotasError.
The output of the failed command is stored in the Schulkonsole::DruckquotasError.

25.6.2012

=cut

require Exporter;
use vars qw(%balance $VERSION @ISA @EXPORT @EXPORT_OK %EXPORT_TAGS);
$VERSION = 0.2;


@ISA = qw(Exporter);
@EXPORT_OK = qw(
	own_quota
	quota

	read_pykota_conf_file
	read_linuxmuster_pykota_conf_file
	get_linuxmuster_pykota_conf
	get_balance
	get_balancelist
	set_balance
	reset_balance
	delete_balance
	get_printers
	add_printer
	delete_printer
	charge_printer
	passthrough_printer
	add_standard
	delete_standard
	modify_standard
	set_aktiv
);

my $wrapcmd = $Schulkonsole::DruckquotasConfig::_wrapper_druckquotas;
my $errorclass = "Schulkonsole::Error::DruckquotasError";


=head2 C<own_quota($id, $password)>

Get print quota of user

=head3 Parameters

=over

=item C<$id>

The ID (not UID) of the user invoking the command

=item C<$password>

The password of the user invoking the command

=back

=head3 Description

This wraps the commands 
C</usr/bin/linuxmuster-pk --user=user -t> 
to get the number of printed pages 
and
C</usr/bin/linuxmuster-pk --user=user -m>
maximum number of pages
with C<user> = the UID of the user with ID C<$id>.

Returns an array with number of printed pages and then the maximum number of 
pages.

=cut

sub own_quota {
	my $id = shift;
	my $password = shift;

	my $in = Schulkonsole::Wrapper::wrap($wrapcmd, $errorclass, Schulkonsole::DruckquotasConfig::PRINTERGETOWNQUOTAAPP,
		$id, $password);

	my ($pages, $max) = $in =~ /^(\d+)\t(\d+)$/;

	die new Schulkonsole::Error::DruckquotasError(
		Schulkonsole::Error::DruckquotasError::WRAPPER_UNEXPECTED_DATA,
		$Schulkonsole::DruckquotasConfig::_wrapper_druckquotas,
	    $in)
		unless defined $max;

	return ($pages, $max);
}





=head2 C<quota($id, $password, @users)>

Get print quota of users

=head3 Parameters

=over

=item C<$id>

The ID (not UID) of the user invoking the command

=item C<$password>

The password of the user invoking the command

=item C<@users>

List of UIDs to get quota from

=back

=head3 Description

This wraps the commands 
C</usr/bin/linuxmuster-pk --user=user -t> 
to get the number of printed pages 
and
C</usr/bin/linuxmuster-pk --user=user -m>
maximum number of pages
invoked for each UID from C<@users> as C<user> 

=head4 Return value

A reference to a hash of the form C<< $user => quota >>, where C<$user> are 
the users of C<@users> and C<quota> is a hash with the keys C<usage> for the 
number of printed pages and C<limit> for the maximum allowed number of 
printed pages.

=cut

sub quota {
	my $id = shift;
	my $password = shift;
	my @users = @_;


	return {} unless @users;


	my @in = Schulkonsole::Wrapper::wrap($wrapcmd, $errorclass, Schulkonsole::DruckquotasConfig::PRINTERGETQUOTAAPP,
		$id, $password, join("\n", @users) . "\n\n", Schulkonsole::Wrapper::MODE_LINES);

	my %re;
	my $buffer;
	foreach (@in) {
		$buffer .= $_;
		my ($user, $pages, $max) = /^(.+)\t(\d+)\t(\d+)$/;

		die new Schulkonsole::Error::DruckquotasError(
			Schulkonsole::Error::DruckquotasError::WRAPPER_UNEXPECTED_DATA,
			$Schulkonsole::DruckquotasConfig::_wrapper_druckquotas,
			$buffer)
			unless defined $max;


		$re{$user} = {
			usage => $pages,
			limit => $max,
		};
	}

	return \%re;
}



sub set_balance {
	my $id = shift;
	my $password = shift;
	my $balance = shift;
	my @benutzer_array = @_;

	Schulkonsole::Wrapper::wrapcommand($wrapcmd, $errorclass, Schulkonsole::DruckquotasConfig::PRINTQUOTAAPP,
		$id, $password, "3\n$balance\n". join("\n", @benutzer_array). "\n\n");
}

sub reset_balance {
	my $id = shift;
	my $password = shift;
	my $balance = shift;
	my @benutzer_array = @_;

	Schulkonsole::Wrapper::wrapcommand($wrapcmd, $errorclass, Schulkonsole::DruckquotasConfig::PRINTQUOTAAPP,
		$id, $password, "4\n$balance\n". join("\n", @benutzer_array). "\n\n");
}

sub delete_balance {
	my $id = shift;
	my $password = shift;
	my @benutzer_array = @_;

	Schulkonsole::Wrapper::wrapcommand($wrapcmd, $errorclass, Schulkonsole::DruckquotasConfig::PRINTQUOTAAPP,
		$id, $password, "2\n". join("\n", @benutzer_array). "\n\n");
}

sub get_balance {
	my $id = shift;
	my $password = shift;
	my $login = shift;

	my $in = Schulkonsole::Wrapper::wrap($wrapcmd, $errorclass, Schulkonsole::DruckquotasConfig::PRINTQUOTASHOWAPP,
		$id, $password, "$login\n". "\n\n");
	my $maxmb,
	my $mb;
	foreach (split('\R', $in)) {
		($maxmb,$mb) = split(/ /, $_);
	}
	return ($maxmb, $mb);
}

sub get_balancelist {
	my $id = shift;
	my $password = shift;
	my @login_array = @_;

	my $in = Schulkonsole::Wrapper::wrap($wrapcmd, $errorclass, Schulkonsole::DruckquotasConfig::PRINTQUOTAAPP,
		$id, $password, "1\n". join("\n", @login_array). "\n\n");

	my @re;
	foreach (split('\R', $in)) {
		last if /^$/;
		push @re, $_;
	}
	
	return @re;
}

sub get_printers {
	my $id = shift;
	my $password = shift;

	my $in = Schulkonsole::Wrapper::wrap($wrapcmd, $errorclass, Schulkonsole::DruckquotasConfig::PRINTQUOTAPRINTERAPP,
		$id, $password, "0\n". "\n\n");

	my @re;
	foreach (split('\R', $in)) {
		last if /^$/;
		push @re, $_;
	}

	return @re;
}

sub add_printer {
	my $id = shift;
	my $password = shift;
	my $printer = shift;

	Schulkonsole::Wrapper::wrapcommand($wrapcmd, $errorclass, Schulkonsole::DruckquotasConfig::PRINTQUOTAPRINTERAPP,
		$id, $password, "1\n$printer\n". "\n\n");
}

sub delete_printer {
	my $id = shift;
	my $password = shift;
	my $printer = shift;

	Schulkonsole::Wrapper::wrapcommand($wrapcmd, $errorclass, Schulkonsole::DruckquotasConfig::PRINTQUOTAPRINTERAPP,
		$id, $password, "2\n$printer\n". "\n\n");
}

sub charge_printer {
	my $id = shift;
	my $password = shift;
	my $value = shift;
	my $printer = shift;

	Schulkonsole::Wrapper::wrapcommand($wrapcmd, $errorclass, Schulkonsole::DruckquotasConfig::PRINTQUOTAPRINTERAPP,
		$id, $password, "3\n$value\n$printer\n". "\n\n");
}

sub passthrough_printer {
	my $id = shift;
	my $password = shift;
	my $value = shift;
	my $printer = shift;

	Schulkonsole::Wrapper::wrapcommand($wrapcmd, $errorclass, Schulkonsole::DruckquotasConfig::PRINTQUOTAPRINTERAPP,
		$id, $password, "4\n$value\n$printer\n". "\n\n");
}

sub get_linuxmuster_pykota_conf {
	my $id = shift;
	my $password = shift;
	do "$Schulkonsole::DruckquotasConfig::_linuxmuster_pykota_conf_file";
	return (%balance);
}

sub read_linuxmuster_pykota_conf_file {
	my $id = shift;
	my $password = shift;
	my @lines;
	if (open PYKOTACONF , $Schulkonsole::DruckquotasConfig::_linuxmuster_pykota_conf_file) {
		while (my $line = <PYKOTACONF> ){
			push (@lines, $line);
			}
		close PYKOTACONF;
	}
	return (@lines);
}

sub read_pykota_conf_file {
	my $id = shift;
	my $password = shift;
	my @lines;
	if (open PYKOTACONF , $Schulkonsole::DruckquotasConfig::_pykota_conf_file) {
		while (my $line = <PYKOTACONF> ){
			push @lines, $line;
		}
		close PYKOTACONF;
	}
	return (@lines);
}

sub add_standard {
	my $id = shift;
	my $password = shift;
	my $group = shift;
	my $value = shift;

	Schulkonsole::Wrapper::wrapcommand($wrapcmd, $errorclass, Schulkonsole::DruckquotasConfig::PRINTQUOTADEFAULTSAPP,
		$id, $password, "1\n$group\n$value\n". "\n\n");
}

sub delete_standard {
	my $id = shift;
	my $password = shift;
	my $group = shift;

	Schulkonsole::Wrapper::wrapcommand(Schulkonsole::DruckquotasConfig::PRINTQUOTADEFAULTSAPP,
		$id, $password, "2\n$group\n". "\n\n");
}

sub modify_standard {
	my $id = shift;
	my $password = shift;
	my $group = shift;
	my $value = shift;

	Schulkonsole::Wrapper::wrapcommand($wrapcmd, $errorclass, Schulkonsole::DruckquotasConfig::PRINTQUOTADEFAULTSAPP,
		$id, $password, "3\n$group\n$value\n". "\n\n");
}

sub set_aktiv {
	my $id = shift;
	my $password = shift;
	my $nummer = shift;

	Schulkonsole::Wrapper::wrapcommand($wrapcmd, $errorclass, Schulkonsole::DruckquotasConfig::PRINTQUOTADEFAULTSAPP,
		$id, $password, "$nummer\n". "\n\n");
}

1;
